# Some of the functions used were taken from: https://eu.udacity.com/course/self-driving-car-engineer-nanodegree--nd013
# The implemented functions have been modified and improved in order to perform some necessary tasks for the purpose of the thesis

# Import libraries
import cv2
import numpy as np
import matplotlib.pyplot as plt
import matplotlib.image as mpimg
import glob
import os
import sys
import warnings
import timeit
import imageio
import datetime
import argparse
from moviepy.editor import VideoFileClip

prev_warped = None   # previous warped image
prev_ret = None      # previous information
processed_frames = 0 # counter of frames processed (when processing video)
bad_frame = 0        # counter of bad frames (not correctly detected)

def process_image(image):
    """
    ## Apply each process steps on a video frame
    """
    
    global prev_warped
    global prev_ret
    global processed_frames
    global bad_frame
    
    # Apply Gradient thresholds
    gradient_combined = apply_thresholds(image, '.jpg', verbose=False)

    # Apply Color thresholds
    comb_binary = apply_color_threshold(image, '.jpg', verbose=False)

    # Combine Gradient and Color binary images
    combined_binary = apply_combined_threshold(comb_binary, gradient_combined)

    # Apply the Perspective Transformation
    binary_warped, Minv = binary_transform(combined_binary, '.jpg', verbose=False)
    
    # Get the Histogram
    histogram = get_histogram(binary_warped)
    
    # Plot and Save histogram
    # plot_histo(histogram, '.jpg')
    
    # Apply the Sliding Window method to detect lane lines
    ploty, left_fit, right_fit = sliding_window(binary_warped, histogram, '.jpg', verbose=False)

    # Skipping Sliding Window
    ret = skip_sliding_window(binary_warped, left_fit, right_fit, '.jpg', verbose=False)
    
    # Measure curvature and vehicle's offset
    curve_diff, avg_curverad = measure_curvature(ploty, ret)
    offset, direction = measure_vehicle_offset(image, ret)
    
    left_fitx = ret['left_fitx']
    right_fitx = ret['right_fitx']
    
    # Sanity check: whether the lines are roughly parallel and have similar curvature
    slope_left = left_fitx[0] - left_fitx[-1]
    slope_right = right_fitx[0] - right_fitx[-1]
    slope_diff = abs(slope_left - slope_right)
    slope_threshold = 150
    curve_threshold = 10000
    
    if processed_frames > 0:
        # if sanity check doesn't go well use the last detected frame
        if (slope_diff > slope_threshold or curve_diff > curve_threshold):
            bad_frame += 1
            binary_warped = prev_warped
            ret = prev_ret
    
    # Visualizing Lane Lines Info
    result = draw_lane_lines(image, binary_warped, Minv, ret)
    
    # Text coordinates  
    org = (30, 60)
    # Font Type
    fontType = cv2.FONT_HERSHEY_DUPLEX
    # Font Scale 
    fontScale = 1.5
    # RED color in RGB 
    RED = (255, 0, 0) 
    # Line thickness of 3 px 
    thickness = 3
    # Text to display
    # if the average curvature is high than a threshold the curve can be approximated as a straigh line
    if avg_curverad < 3000:
        curvature_text = 'Average Radius of curvature = ' + str(round(avg_curverad, 3)) + 'm'
    else:
        curvature_text = 'Straight'
    
    # Annotate radius of curvature on the screen
    cv2.putText(result, curvature_text, org, fontType, fontScale, RED, thickness)
    
    # Annotate vehicle's offset on the screen
    offset_text = 'Vehicle is ' + str(round(abs(offset), 3)) + 'm ' + direction + ' of center'
    cv2.putText(result, offset_text, (org[0], org[1]+50), fontType, fontScale, RED, thickness)

    prev_warped = binary_warped
    prev_ret = ret
    
    processed_frames += 1
    
    return result

def usage():
    print ('Usage: '+ sys.argv[0] + ' <image_file> or <video_file>')
    print("Example: python ALLD.py -i=input_image.png (or -i=input_video.avi)")
    
def save(filename, image, file_extension):
    """
    ## Function used to save the image on device preserving the format extension using OpenCV library
    """
        
    # Saving the image
    isWritten = cv2.imwrite(filename + file_extension, image)
    if isWritten:
        print(filename + " Image successfully saved!")
    
    return

def save_files(image1, image2, image3, image4, file_extension, image1_name="Image 1", image2_name="Image 2", image3_name="Image 3", image4_name="Image 4", flag=False):
    """
    ## Function used to save the images on device preserving the format extension using matplotlib library
    """
    
    if(flag==True):
        color='viridis' # default colormap
    else:
        color='gray'

    mpimg.imsave(image1_name + file_extension, image1, cmap=color)
    mpimg.imsave(image2_name + file_extension, image2, cmap=color)
    mpimg.imsave(image3_name + file_extension, image3, cmap=color)
    mpimg.imsave(image4_name + file_extension, image4, cmap=color)
    
    return

def compare_four_images(image1, image2, image3, image4, image1_exp="Image 1", image2_exp="Image 2", image3_exp="Image 3", image4_exp="Image 4", flag=False):
    """
    ## Function used to display 4 different images
    """
    
    f, ((ax1, ax2), (ax3, ax4)) = plt.subplots(2, 2, figsize=(12.80,7.20))
    
    if(flag==True):
        color='viridis' # default colormap
    else:
        color='gray'
    
    ax1.set_title(image1_exp, fontsize=10)
    ax1.imshow(image1, cmap=color)
    ax2.set_title(image2_exp, fontsize=10)
    ax2.imshow(image2, cmap=color)
    ax3.set_title(image3_exp, fontsize=10)
    ax3.imshow(image3, cmap=color)
    ax4.set_title(image4_exp, fontsize=10)
    ax4.imshow(image4, cmap=color)
    
    return

def compare_images(image1, image2, image1_exp="Image 1", image2_exp="Image 2", flag=False):
    """
    ## Function used to display 2 different images
    """
    
    if(flag==True):
        color='viridis' # default colormap
    else:
        color='gray'
    
    f, (ax1, ax2) = plt.subplots(1, 2, figsize=(12.80,7.20))
    f.tight_layout()
    ax1.set_title(image1_exp, fontsize=30)
    ax1.imshow(image1)
    ax2.set_title(image2_exp, fontsize=30)
    ax2.imshow(image2, cmap=color)
    plt.show()
    
    return

def abs_sobel_threshold(image, orient='x', sobel_kernel=3, thresh=(0, 255)):
    """
    # The following steps are applied to image in order to take either the x or y gradient using Sobel operator
    ## 1) Convert the image to grayscale
    ## 2) Take the derivative in x or y given orient = 'x' or 'y' using Sobel operator
    ## 3) Take the absolute value of the derivative or gradient
    ## 4) Scale to 8-bit (0 - 255) then convert to type = np.uint8
    ## 5) Create a mask of 1's where the scaled gradient magnitude
    ##    is >= thresh_min and <= thresh_max (even the exclusive brackets are fine)
    ## 6) Return this mask as grad_binary output image
    """ 
    
    gray = cv2.cvtColor(image, cv2.COLOR_RGB2GRAY)
    isX = True if orient == 'x' else False # orient =='y'
    depth = cv2.CV_64F
    sobel = cv2.Sobel(gray, depth, isX, not isX)
    abs_sobel = np.absolute(sobel)
    scaled_sobel = np.uint8(255*abs_sobel/np.max(abs_sobel))
    # create a mask copy of scaled_sobel and apply threshold
    grad_binary = np.zeros_like(scaled_sobel)
    # create a binary threshold to select pixels based on gradient strength
    # the output array elements is 1 where the gradients are in the threshold range, and 0 everywhere else
    grad_binary[(scaled_sobel >= thresh[0]) & (scaled_sobel <= thresh[1])] = 1
    
    return grad_binary

def magnitude_threshold(image, sobel_kernel=3, mag_thresh=(0, 255)):
    """
    # The following steps are applied to image in order to take the magnitude of the gradient
    ## 1) Convert to grayscale
    ## 2) Take the gradient in x and y separately
    ## 3) Calculate the magnitude
    ## 4) Scale to 8-bit (0 - 255) and convert to type = np.uint8
    ## 5) Create a binary mask where mag thresholds are met
    ## 6) Return this mask as mag_binary output image
    """ 
    
    gray = cv2.cvtColor(image, cv2.COLOR_RGB2GRAY)
    depth = cv2.CV_64F
    sobelx = cv2.Sobel(gray, depth, 1, 0, ksize=sobel_kernel)
    sobely = cv2.Sobel(gray, depth, 0, 1, ksize=sobel_kernel)
    abs_sobel = np.sqrt(sobelx**2 + sobely**2)
    # Rescale to 8-bit
    gradmag = np.uint8(255*abs_sobel/np.max(abs_sobel))
    # Create a binary image of 1s where threshold is met, 0s otherwise
    mag_binary = np.zeros_like(gradmag)
    mag_binary[(gradmag >= mag_thresh[0]) & (gradmag <= mag_thresh[1])] = 1

    return mag_binary

def direction_threshold(image, sobel_kernel=3, thresh=(0, np.pi/2)):
    """
    # The following steps are applied to image in order to take the direction of the gradient
    ## 1) Convert to grayscale
    ## 2) Take the gradient in x and y separately
    ## 3) Take the absolute value of the x and y gradients
    ## 4) Calculate the direction of the gradient
    ## 5) Create a binary mask where direction thresholds are met
    ## 6) Return this mask as dir_binary output image
    """ 
    
    gray = cv2.cvtColor(image, cv2.COLOR_RGB2GRAY)
    depth = cv2.CV_64F
    sobelx = cv2.Sobel(gray, depth, 1, 0, ksize=sobel_kernel)
    sobely = cv2.Sobel(gray, depth, 0, 1, ksize=sobel_kernel)
    abs_sobelx = np.absolute(sobelx)
    abs_sobely = np.absolute(sobely)
    grad_dir = np.arctan2(abs_sobely, abs_sobelx)
    # apply a threshold, and create a binary image result
    dir_binary = np.zeros_like(grad_dir)
    dir_binary[(grad_dir >= thresh[0]) & (grad_dir <= thresh[1])] = 1

    return dir_binary

def apply_thresholds(image, file_extension, verbose=True, ksize=3):
    """
    # The following steps are applied to image in order to obtain binary result from multiple thresholds
    ## 1) Take the gradient in x and y with magnitude and direction
    ## 2) Combining Thresholds of point 1
    """ 
    
    gradx = abs_sobel_threshold(image, orient='x', sobel_kernel=ksize, thresh=(20, 100))
    grady = abs_sobel_threshold(image, orient='y', sobel_kernel=ksize, thresh=(20, 100))
    
    mag_binary = magnitude_threshold(image, sobel_kernel=ksize, mag_thresh=(30, 100))
    dir_binary = direction_threshold(image, sobel_kernel=ksize, thresh=(0.7, 1.3))
    
    # Display and Saving the thresholding images
    
    if verbose:
        save_files(gradx, grady, mag_binary, dir_binary, file_extension, 'Gradient_x', 'Gradient_y', 'Gradient Magnitude','Gradient Direction', True)
    
        compare_four_images(gradx, grady, mag_binary, dir_binary, "Gradient x", "Gradient y", "Gradient Magnitude", "Gradient Direction", False)
        plt.suptitle('Gradient Thresholds', fontsize=16)
    
    # Combining Thresholds: binary result from multiple thresholds
    # selection of pixels where both the x and y gradients meet the threshold criteria, or the gradient magnitude and direction are both within their threshold values
    
    combined = np.zeros_like(dir_binary)
    combined[((gradx == 1) & (grady == 1)) | ((mag_binary == 1) & (dir_binary == 1))] = 1

    return combined

def apply_color_threshold(image, file_extension, verbose=True):
    """
    ## This function convert an RGB image to 4 color spaces: RGB, HLS, LAB and LUV in order to
    ## pick up the lane lines, then apply different thresholds on:
	    -the R channel of RGB color space,
        -the S channel of HLS color space,
	    -the B channel of LAB color space,
        -the L channel of LUV color space,
    ## Creating binary images with threshold it is possible to correctly isolate the pixels of the lane line
	## using appropriate channels (the best ones) of different color spaces.
	## Finally, combine them together in a binary output image of threshold result
    """
    
    # convert RGB image to HLS color space (Hue Light Saturation)
    hls = cv2.cvtColor(image, cv2.COLOR_RGB2HLS)
    # convert RGB image to LUV color space
    luv = cv2.cvtColor(image, cv2.COLOR_RGB2LUV)
    # convert RGB image to LAB color space
    Lab = cv2.cvtColor(image, cv2.COLOR_RGB2Lab)
    
    ## isolate each color channels from each color spaces and display them
    
    R_channel = image[:,:,0]
    G_channel = image[:,:,1]
    B_channel = image[:,:,2]
    
    # compare_four_images(image, R_channel, G_channel, B_channel, "Original Image", "R channel", "G channel", "B channel")
    # plt.suptitle('RGB Color Space channels', fontsize=16)
    
    h_channel = hls[:,:,0]
    l1_channel = hls[:,:,1]
    s_channel = hls[:,:,2]
    
    # compare_four_images(image, h_channel, l1_channel, s_channel, "Original Image", "H channel", "L channel", "S channel")
    # plt.suptitle('HLS Color Space channels', fontsize=16)
     
    l_channel = luv[:,:,0]
    u_channel = luv[:,:,1]
    v_channel = luv[:,:,2]
    
    # compare_four_images(image, l_channel, u_channel, v_channel, "Original Image", "L channel", "U channel", "V channel")
    # plt.suptitle('LUV Color Space channels', fontsize=16)
    
    L_channel = Lab[:,:,0]
    a_channel = Lab[:,:,1]
    b_channel = Lab[:,:,2]
    
    # compare_four_images(image, L_channel, a_channel, b_channel, "Original Image", "L channel", "A channel", "B channel")
    # plt.suptitle('LAB Color Space channels', fontsize=16)
    
    # Create binary images with threshold to isolate the pixels of the lane line using
    # appropriate channels (the best ones) of different color spaces
    
    # binary threshold of R channel
    r_thresh_min=135
    r_thresh_max=255
    r_binary = np.zeros_like(R_channel)
    r_binary[(R_channel >= r_thresh_min) & (R_channel <= r_thresh_max)] = 1
    
    # binary threshold of S channel
    s_thresh_min = 70
    s_thresh_max = 255
    s_binary = np.zeros_like(s_channel)
    s_binary[(s_channel >= s_thresh_min) & (s_channel <= s_thresh_max)] = 1
    
    # binary threshold of B channel
    b_thresh_min = 140
    b_thresh_max = 200
    b_binary = np.zeros_like(b_channel)
    b_binary[(b_channel >= b_thresh_min) & (b_channel <= b_thresh_max)] = 1
    
    # binary threshold of L channel
    l_thresh_min = 140
    l_thresh_max = 255
    l_binary = np.zeros_like(l_channel)
    l_binary[(l_channel >= l_thresh_min) & (l_channel <= l_thresh_max)] = 1
    
    # Display and Save the binary thresholded images
    if verbose:
        compare_four_images(r_binary, s_binary, b_binary, l_binary, "RGB-R threshold", "HLS-S threshold", "LAB-B threshold", "LUV-L threshold", False)
        plt.suptitle('Color Thresholds', fontsize=16)
        save_files(r_binary, s_binary, b_binary, l_binary, file_extension, "R binary", "S binary", "B binary", "L binary", False)
    
    # Combine thresholded binary images
    combined_binary = np.zeros_like(s_binary)
    combined_binary[(l_binary == 1) | (b_binary == 1) | (r_binary == 1) | (s_binary == 1)] = 1
    
    # Show the binary combined color threshold
    # plt.figure()
    # plt.imshow(combined_binary, cmap='gray')
    # plt.suptitle('Combined Color thresholds', fontsize=16)
    # plt.show()
    
    return combined_binary

def apply_combined_threshold(c_binary, combined):
    """
    ## Function that combine the thresholded binary color with the thresholded binary gradient returning as output a binary image
    """
    
    combined_binary = np.zeros_like(combined)
    combined_binary[(c_binary == 1) | (combined == 1)] = 1

    return combined_binary

def ROI(image, points, file_extension, save=False):
    """
    ## Function used to show the ROI on the image
    """
        
    # Let's define four points
    pts = np.array( points, np.int32)
    # Let's now reshape our points in form required by polylines
    pts = pts.reshape((-1,1,2))
    # Build the polygon with the blue color
    cv2.polylines(image, [pts], True, (0,0,255), 3)
    
    # Show the ROI
    img = cv2.cvtColor(image, cv2.COLOR_RGB2BGR)
    cv2.imshow('ROI',img)
    cv2.waitKey(0)
    cv2.destroyAllWindows()
    
    # Save the ROI
    if(save==True):
        name = 'ROI' + file_extension
        isWritten = cv2.imwrite(name, img)
        if isWritten:
            print("ROI Image successfully saved!")
    
    return

def binary_transform(img, file_extension, verbose=True):
    """
    ## This function performs the task of deforming the binary image so that it is viewed from above (bird's eye view).
    ## To do this, 2 sets of points are used to also calculating Transformation Matrix: 4 coordinates of source points to set the ROI in the original image
    ## and 4 coordinates of destination points to map them in the warped image.
    """
    
    img_size = (img.shape[1], img.shape[0]) # width x height
    # print(img_size)
    h, w = img.shape[:2]
    
    Left_Bottom=[255, h]    # left bottom most point of trapezium
    Right_Bottom=[1037, h]  # right bottom most point of trapezium
    Left_Top=[605, 395]     # left top most point of trapezium
    Right_Top=[735, 395]    # right top most point of trapezium
    
    src=np.float32([Left_Bottom, Left_Top, Right_Top, Right_Bottom]) # Source Points
    
    # Let's check out ROI on the orignal image using the source points
    if verbose:
        if(len(img.shape)==3):
            image = img.copy()
            ROI(image, src, file_extension, True)
        
    # Destination Points
    dst = np.float32(
        [   [250, h],   # left bottom
            [250, 0],   # left top
            [900, 0],   # right top
            [900, h]    # right bottom
          ]) 

    # Given src and dst points, calculate the perspective transform matrix and its inverse
    M = cv2.getPerspectiveTransform(src, dst)
    Minv = cv2.getPerspectiveTransform(dst, src)
    
    # Warp the image (the destination image has the same dimensions as the original image)
    binary_warped = cv2.warpPerspective(img, M, img_size, flags=cv2.INTER_LINEAR)
    
    # Let's check out ROI on the warped image using the destination points
    if verbose:
        if(len(img.shape)==3):
            image = binary_warped.copy()
            ROI(image, dst, file_extension, False)
    
    return binary_warped, Minv

def plot_histo(histogram, file_extension):
    """
    ## Plot and Save the histogram using matplotlib
    """
    
    plt.plot(histogram, linestyle = '-',color = '#d41100', linewidth = 2)
    plt.title('Histogram Peaks')
    plt.grid(True)
    plt.xlabel('Pixel Positions')
    plt.ylabel('Counts')
    plt.savefig('Histogram_Peaks' + file_extension)
    plt.show()
    
    return

def get_histogram(binary_warped):
    """
    ### This function is used to identify the peaks in the histogram of the binary image in order to determine the position of the lane lines
    """
    
    # Grab only the bottom half of the image
    # Lane lines are likely to be mostly vertical nearest to the car
    bottom_half = binary_warped[binary_warped.shape[0]//2:,:]
    
    # Sum across image pixels vertically in such a way that the highest areas of vertical lines should be larger values
    histogram = np.sum(bottom_half,axis=0)
    
    # or just in one line: histogram = np.sum(binary_warped[binary_warped.shape[0]//2:,:], axis=0)
    
    return histogram

def display_poly(out_image, left_fitx, right_fitx, ploty, s, file_extension):
    """
    ## This function is used to display the fitted polynomial lines using matplotlib
    """
    
    # Create output image
    out_image = out_image.astype(np.uint8)
    plt.imshow(out_image)
    plt.xlim(0, 1280)
    plt.ylim(720, 0)
    # plots the left and right polynomials on the lane lines
    plt.plot(left_fitx, ploty, color='yellow',linewidth=3)
    plt.plot(right_fitx, ploty, color='yellow',linewidth=3)
    if(s==0):
        plt.suptitle('Sliding Window', fontsize=16)
        plt.savefig('Sliding Window'+ file_extension)
    else:
        plt.suptitle('Skipping Sliding Window', fontsize=16)
        plt.savefig('Skipping Sliding Window' + file_extension)
    plt.show()
    
    # mpimg.imsave('Sliding Window' + file_extension, out_image)
    
    return

def sliding_window(binary_warped, histogram, file_extension, verbose=True):
    """
    ## This function apply a slide window method both for the left and the right lane line, in order to follow accurately the lane lines pixels
    """
    
    global processed_frames
    # Create an output image to draw on and visualize the result
    out_img = np.dstack((binary_warped, binary_warped, binary_warped))*255
    
    # Find the peak of the left and right halves of the histogram, these will be the starting point for the left and right lines
    midpoint = np.int(histogram.shape[0]/2)
    # This is used to create a windows around these maxima
    leftx_base = np.argmax(histogram[:midpoint])
    rightx_base = np.argmax(histogram[midpoint:]) + midpoint 
    
    # Show these 3 values to know lane lines initial position (leftx_base and rightx_base)
    # print('midpoint: ', midpoint, 'leftx_base: ', leftx_base, 'rightx_base: ', rightx_base)
    
    ### set the appearance of the windows (window settings)
    # Choose the number of sliding windows
    nwindows = 12
    # Set the width of the windows +/- margin
    margin = 100 # How much to slide left and right for searching
    # Set the minimum number of pixels found to recenter window
    minpix = 50
    
    # Set the height of windows based on nwindows and image shape
    window_height = np.int(binary_warped.shape[0]/nwindows)
    # Identify the x and y positions of all nonzero (i.e. activated) pixels in the image
    nonzero = binary_warped.nonzero()
    nonzeroy = np.array(nonzero[0])
    nonzerox = np.array(nonzero[1])
    # Current positions to be updated later for each window in nwindows
    leftx_current = leftx_base
    rightx_current = rightx_base
    # Create empty lists to receive left and right lane pixel indices
    left_lane_inds = []
    right_lane_inds = []
    
    # Iterate through each window in nwindows to track curvature using for loop

    for window in range(nwindows):
        # Identify window boundaries in x and y (for right and left)
        win_y_low = binary_warped.shape[0] - (window+1)*window_height
        win_y_high = binary_warped.shape[0] - window*window_height
        win_xleft_low = leftx_current - margin
        win_xleft_high = leftx_current + margin
        win_xright_low = rightx_current - margin
        win_xright_high = rightx_current + margin
        # Draw the windows on the visualization image
        cv2.rectangle(out_img,(win_xleft_low,win_y_low),(win_xleft_high,win_y_high), (0,255,0), 2)
        cv2.rectangle(out_img,(win_xright_low,win_y_low),(win_xright_high,win_y_high), (0,255,0), 2)
        # Identify the nonzero pixels in x and y within the window
        good_left_inds = ((nonzeroy >= win_y_low) & (nonzeroy < win_y_high) &
        (nonzerox >= win_xleft_low) &  (nonzerox < win_xleft_high)).nonzero()[0]
        good_right_inds = ((nonzeroy >= win_y_low) & (nonzeroy < win_y_high) &
        (nonzerox >= win_xright_low) &  (nonzerox < win_xright_high)).nonzero()[0]
        # Append these indices to the lists
        left_lane_inds.append(good_left_inds)
        right_lane_inds.append(good_right_inds)
        # If you found > minpix pixels, recenter next window (leftx_current or rightx_current) on their mean position
        if len(good_left_inds) > minpix:
            leftx_current = np.int(np.mean(nonzerox[good_left_inds]))
        if len(good_right_inds) > minpix:        
            rightx_current = np.int(np.mean(nonzerox[good_right_inds]))
    
    # Concatenate the arrays of indices
    try:
        left_lane_inds = np.concatenate(left_lane_inds)
        right_lane_inds = np.concatenate(right_lane_inds)
    # Avoids an error if the above is not implemented fully
    except ValueError:
        pass
    
    if(left_lane_inds.size == 0):
        print('in frame: ', processed_frames, '-> Left lane line not correctly detected!')
    if(right_lane_inds.size == 0):
        print('in frame: ', processed_frames, '-> Right lane line not correctly detected!')
        
    # Extract left and right line pixel positions
    leftx = nonzerox[left_lane_inds]
    lefty = nonzeroy[left_lane_inds]
    rightx = nonzerox[right_lane_inds]
    righty = nonzeroy[right_lane_inds]
    
    # Fit a second order polynomial to each line pixel positions
    order = 2 # order
    
    if len(leftx) == 0:
        left_fit =[]
    else:
        left_fit = np.polyfit(lefty, leftx, order)
    if len(rightx) == 0:
        right_fit =[]
    else:
        right_fit = np.polyfit(righty, rightx, order)
    
    # x and y values is generated for plotting
    ploty = np.linspace(0, binary_warped.shape[0]-1, binary_warped.shape[0])
    try:
        left_fitx = left_fit[0]*ploty**order + left_fit[1]*ploty + left_fit[2]
        right_fitx = right_fit[0]*ploty**order + right_fit[1]*ploty + right_fit[2]
    # Avoids an error if left_fit and right_fit are still none or incorrect
    except TypeError:
        print("Failed to fit a line!\n")
        left_fitx = 1*ploty**order + 1*ploty
        right_fitx = 1*ploty**order + 1*ploty
        
    # Visualization:
    # Colors in the left and right lane regions
    out_img[lefty, leftx] = [255, 0, 0] # RED
    out_img[righty, rightx] = [0, 0, 255] # BLUE
    
    # Display and save the output image
    if verbose:
        display_poly(out_img, left_fitx, right_fitx, ploty, 0, file_extension)
    
    return ploty, left_fit, right_fit

def skip_sliding_window(binary_warped, left_fit, right_fit, file_extension, verbose=True):
    """
    ## This function fit a polynomial to all the relevant pixels founded in sliding windows method,
    ## and set the area to search for acrivated pixels based on margin out from the fit polynomial
    """
    
    # Width of the margin around the previous polynomial to search
    margin = 100
    # Grab activated pixels
    nonzero = binary_warped.nonzero()
    nonzeroy = np.array(nonzero[0])
    nonzerox = np.array(nonzero[1])
    
    # Set the area of search based on activated x-values
    # within the +/- margin of polynomial function
    
    left_lane_inds = ((nonzerox > (left_fit[0]*(nonzeroy**2) + left_fit[1]*nonzeroy +
                    left_fit[2] - margin)) & (nonzerox < (left_fit[0]*(nonzeroy**2) +
                    left_fit[1]*nonzeroy + left_fit[2] + margin)))

    right_lane_inds = ((nonzerox > (right_fit[0]*(nonzeroy**2) + right_fit[1]*nonzeroy +
                    right_fit[2] - margin)) & (nonzerox < (right_fit[0]*(nonzeroy**2) +
                    right_fit[1]*nonzeroy + right_fit[2] + margin)))  
    
    # Extract left and right line pixels positions
    leftx = nonzerox[left_lane_inds]
    lefty = nonzeroy[left_lane_inds]
    rightx = nonzerox[right_lane_inds]
    righty = nonzeroy[right_lane_inds]
    
    # Fit a second order polynomial to each
    order = 2 # order
    left_fit = np.polyfit(lefty, leftx, order)
    right_fit = np.polyfit(righty, rightx, order)
    # Generate x and y values for plotting
    ploty = np.linspace(0, binary_warped.shape[0]-1, binary_warped.shape[0]) # to cover same y-range as image
    # Calculate both polynomials using ploty, left_fit and right_fit
    left_fitx = left_fit[0]*ploty**order + left_fit[1]*ploty + left_fit[2]
    right_fitx = right_fit[0]*ploty**order + right_fit[1]*ploty + right_fit[2]

    ## Visualization
    
    # Create a new image to draw on and an image to show the selection window
    out_img = np.dstack((binary_warped, binary_warped, binary_warped))*255
    
    window_img = np.zeros_like(out_img)
    out_img[lefty, leftx] = [255, 0, 0] # RED
    out_img[righty, rightx] = [0, 0, 255] # BLUE
  
    # Generate a polygon to illustrate the search window area for left and right lines
    # and recast the x and y points into usable format for cv2.fillPoly()
    # Stack arrays in sequence vertically (row wise)
    left_line_window1 = np.array([np.transpose(np.vstack([left_fitx-margin, ploty]))])
    left_line_window2 = np.array([np.flipud(np.transpose(np.vstack([left_fitx+margin, ploty])))])
    
    # Stack left window arrays in sequence horizontally (column wise)
    left_line_pts = np.hstack((left_line_window1, left_line_window2))
    
    right_line_window1 = np.array([np.transpose(np.vstack([right_fitx-margin, ploty]))])
    right_line_window2 = np.array([np.flipud(np.transpose(np.vstack([right_fitx+margin, ploty])))])
    
    # Stack right windows arrays in sequence horizontally (column wise)
    right_line_pts = np.hstack((right_line_window1, right_line_window2))
    
    # Draw the lane onto the warped blank
    cv2.fillPoly(window_img, np.int_([left_line_pts]), (0,255, 0))
    cv2.fillPoly(window_img, np.int_([right_line_pts]), (0,255, 0))
    result = cv2.addWeighted(out_img, 1, window_img, 0.3, 0)
    
    # Plot and save the polynomial lines onto the image
    if verbose:
        display_poly(result, left_fitx, right_fitx, ploty, 1, file_extension)
    
    # Create a dictionary to return useful data for next steps
    ret = {}
    ret['leftx'] = leftx
    ret['rightx'] = rightx
    ret['left_fitx'] = left_fitx
    ret['right_fitx'] = right_fitx
    ret['ploty'] = ploty

    return ret

def measure_curvature(ploty, lines_info):
    """
    ## Calculates the Radius of curvature R of polynomial functions in meters for each lane line
    """
    
    # Parameters used for the conversion in x and y from pixels space to meters (world space)
    ym_per_pix = 30/720     # meters per pixel in y dimension (lane length)
    xm_per_pix = 3.7/700    # meters per pixel in x dimension (lane width)

    leftx = lines_info['left_fitx']
    rightx = lines_info['right_fitx']

    leftx = leftx[::-1]     # reverse all items in the leftx to match top-to-bottom in y
    rightx = rightx[::-1]   # reverse all items in the rightx to match top-to-bottom in y
    
    # Define y-value where we want radius of curvature
    # Choose the max y-value, correspond to the bottom of the image
    y_eval = np.max(ploty)
    # Fit new polynomials to x, y in world space
    order=2
    left_fit_cr = np.polyfit(ploty*ym_per_pix, leftx*xm_per_pix, order)
    right_fit_cr = np.polyfit(ploty*ym_per_pix, rightx*xm_per_pix, order)
    
    # Calculation of R_curve (Radius of curvature)
    left_curverad = ((1 + (2*left_fit_cr[0]*y_eval*ym_per_pix + left_fit_cr[1])**2)**1.5) / np.absolute(2*left_fit_cr[0])
    right_curverad = ((1 + (2*right_fit_cr[0]*y_eval*ym_per_pix + right_fit_cr[1])**2)**1.5) / np.absolute(2*right_fit_cr[0])
    
    # Calculate the average and the difference between the 2
    diff_curverad = abs(left_curverad - right_curverad)
    avg_curverad = (left_curverad + right_curverad) / 2
    
    # print the above values rounded
    # print('left_curverad: ', round(left_curverad,4), 'm','right_curverad', round(right_curverad,4), 'm','avg_curverad', round(avg_curverad,4), 'm')
    
    return diff_curverad, avg_curverad

def measure_vehicle_offset(image, lines_info):
    """
    ### vehicle's offset calculation, based on 2 assumptions:
    ####        1) camera mounted in the center of vehicle
    ####        2) road lane is 3.5 meters wide (from CARLA Simulator)
    ## Calculates vehicle's offset and its direction in meters
    """
    
    leftx = lines_info['left_fitx']
    rightx = lines_info['right_fitx']

    xm_per_pix = 3.5/700 # meters per pixel in x dimension (lane width)
    
    # Calculate the lane center
    lane_center = (rightx[-1] + leftx[-1])/2
    # print('lane_center: ', lane_center)
    # Calculate the pixel deviation based on the width of image
    camera_center = image.shape[1]/2.0  # The horizontal center of the image
    offset_pixels = (lane_center - camera_center)
    
    # Convert the vehicle's offset from pixels to meters
    offset = offset_pixels * xm_per_pix
    # print the vehicle's offset
    # print('offset: ', offset)
    # Calculates the offset direction of the vehicle
    direction = "left" if offset < 0 else "right"
    
    return offset, direction

def draw_lane_lines(original_image, warped_image, Minv, draw_info):
    """
    ## This function draws and highlights the right and left lane lines
    """
    
    leftx = draw_info['leftx']
    rightx = draw_info['rightx']
    left_fitx = draw_info['left_fitx']
    right_fitx = draw_info['right_fitx']
    ploty = draw_info['ploty']
    
    # Create an image to draw the lines on
    warp_zero = np.zeros_like(warped_image).astype(np.uint8)
    # Stack arrays in sequence depth wise (along third axis)
    color_warp = np.dstack((warp_zero, warp_zero, warp_zero))
    
    # Recast the x and y points into usable format for cv2.fillPoly()
    # Stack arrays in sequence vertically (row wise)
    pts_left = np.array([np.transpose(np.vstack([left_fitx, ploty]))])
    pts_right = np.array([np.flipud(np.transpose(np.vstack([right_fitx, ploty])))])
    # Stack arrays in sequence horizontally (column wise)
    pts = np.hstack((pts_left, pts_right))
    
    # Fills the area bounded by the polygon and draw the lane lines
    # Draw the lane onto the warped blank image
    cv2.fillPoly(color_warp, np.int_([pts]), (0,255, 0))
    # Draw the left and right lines
    cv2.polylines(color_warp, np.int32([pts_left]), isClosed=False, color=(220,0,0), thickness=15)
    cv2.polylines(color_warp, np.int32([pts_right]), isClosed=False, color=(0,50,255), thickness=15)
    
    # Warp the blank back to original image space using inverse perspective matrix (Minv)
    newwarp = cv2.warpPerspective(color_warp, Minv, (original_image.shape[1], original_image.shape[0]))
    # Combine the result with the original image
    result = cv2.addWeighted(original_image, 1, newwarp, 0.3, 0)

    return result

def main(args):
    """
    ## Main function used to apply the Advanced Lane Line Detection to an image or video
    """
    global bad_frame
    
    if len(args) == 1:
        
        pathfile = args['input']
        
        print(f"""The passed parameter is: {pathfile}""")
        
        # Check if the extension belongs the most common List of Formats Supported by OpenCV
        
        if pathfile.lower().endswith(('.png', '.jpg', '.jpeg')):
            
            if not os.path.isfile(pathfile):
                print("Input image file ", pathfile, " doesn't exist")
                usage()
                sys.exit()
                
            filename, file_extension = os.path.splitext(pathfile)
            print('filename: ', filename)
            print('file_extension: ', file_extension)
            
            ### Note: If you read an image using the matplotlib library: matplotlib.image.imread () you get an RGB image,
            ### instead, if you read an image using the OpenCV cv2.imread () library you get a BGR image. Both libraries can be used in combination.
            
            # To Show and Save the intermediate step images change verbose = True
            
            #image = mpimg.imread(pathfile)
            image = cv2.imread(pathfile)
            if image is None:
                    print("Unable to read file. Exiting...")
                    sys.exit()
            # # height, width, number of channels in image
            height = image.shape[0]
            width = image.shape[1]
            channels = image.shape[2]

            print('Image Height       : ',height)
            print('Image Width        : ',width)
            print('Number of Channels : ',channels)
            
            # Get the tick count
            e1 = cv2.getTickCount()
            
            # Resizing the image
            print("Resizing the input image...")
            image = cv2.resize(image, dsize=(1280, 720))
            
            (H, W) = image.shape[:2]
            print('Shape of the resized image: Width x Height:', W, 'x', H)
            
            # Showing the image
            img = cv2.cvtColor(image, cv2.COLOR_BGR2RGB)
            
            cv2.namedWindow("Image", cv2.WINDOW_AUTOSIZE)
            cv2.imshow('Image', image)
            cv2.waitKey(0)
            cv2.destroyAllWindows()
            
            # or using matplotlib:
            # imgplot = plt.imshow(img)
            # #plt.savefig('Original' + file_extension)
            # plt.show()
            
            # Saving the image
            save('Original', image, file_extension)
            
            # Edge Detection - Gradient Thresholds
            combined = apply_thresholds(img, file_extension, verbose=False)
            compare_images(img, combined, "Original Image", "Combined Gradient Thresholds", False)
            
            # Save output
            mpimg.imsave('Combined Binary Gradient' + file_extension, combined, cmap='gray')
            
            # Color Selection - Color Thresholds
            combined_color = apply_color_threshold(img, file_extension, verbose=False)
            compare_images(img, combined_color, "Original Image", "Combined Color Thresholds", False)
            
            # Save output
            mpimg.imsave('Combined Binary Color' + file_extension, combined_color, cmap='gray')
            
            # In order to correctly identify the lane lines a combination of color and gradient thresholds is performed to generate a robust binary image
            combined_binary = apply_combined_threshold(combined_color, combined)
            compare_images(img, combined_binary, "Original Image", "Combined Binary Gradient and Color", False)
            
            mpimg.imsave('Gradient & Color Binary Combined' + file_extension, combined_binary, cmap='gray')
            
            # Perspective transformation of image
            warped, Minv = binary_transform(img, file_extension, verbose=False)
            compare_images(img, warped, "Original Image", "Warped Image", False)
            
            # Save output
            mpimg.imsave('Warped Image' + file_extension, warped)
            
            # Perspective transformation of binary image
            binary_warped, Minv = binary_transform(combined_binary, file_extension, verbose=False)
            
            compare_images(img, binary_warped, "Original Image", "Binary Warped Image")
            
            # Save output
            mpimg.imsave('Binary Warped Image' + file_extension, binary_warped, cmap='gray')
            
            # Get histogram from binary warped image
            histogram = get_histogram(binary_warped)
            
            # Plot and Save histogram
            plot_histo(histogram, file_extension)
    
            # Sliding Window method
            ploty, left_fit, right_fit = sliding_window(binary_warped, histogram, file_extension, verbose=False)
            
            # Skipping Sliding Window method
            draw_info = skip_sliding_window(binary_warped, left_fit, right_fit, file_extension, verbose=False)

            # Measure curvature
            diff_curverad, avg_curverad = measure_curvature(ploty, draw_info)
            
            # Measure vehicle's offset
            offset, direction = measure_vehicle_offset(img, draw_info)
            
            # Plotting the lines and the drive area on the unwarped image (back to original image using inverse perspective transformation -> Minv)
            result = draw_lane_lines(img, binary_warped, Minv, draw_info)
            
            # Annotate curvature on the screen
            # Text coordinates  
            org = (30, 60)
            # Font Type
            fontType = cv2.FONT_HERSHEY_DUPLEX
            # Font Scale 
            fontScale = 1.5
            # RED color in RGB 
            RED = (255, 0, 0) 
            # Line thickness of 3 px 
            thickness = 3
            # Text to display
            # if the average curvature is high than a threshold the curve can be approximated as a straight line
            if avg_curverad < 3000:
                curvature_text = 'Average Radius of curvature = ' + str(round(avg_curverad, 3)) + 'm'
            else:
                curvature_text = 'Straight'
            
            # Put text information
            cv2.putText(result, curvature_text, org, fontType, fontScale, RED, thickness)
            
            # Annotate vehicle's offset on the screen
            offset_text = 'Vehicle is ' + str(round(abs(offset), 3)) + 'm ' + direction + ' of center'
            cv2.putText(result, offset_text, (org[0], org[1]+50), fontType, fontScale, RED, thickness)
            
            # Print the processing time -> to obtain a true value, disable any intermediate display
            e2 = cv2.getTickCount()
            time = (e2 - e1)/cv2.getTickFrequency()
            print('Total Processing Time: ', time, 'seconds')
            
            # Show the final result
            final = cv2.cvtColor(result, cv2.COLOR_RGB2BGR)
            cv2.imshow('FINAL', final)
            cv2.waitKey(0)
            cv2.destroyAllWindows()
            
            save('Final', final, file_extension)
            
            # or using matplotlib:
            # plt.imshow(result)
            # plt.axis('off')
            # mpimg.imsave('FINAL'+ file_extension, result)
            # plt.show()
            
        elif pathfile.lower().endswith(('.avi', '.mp4')): # most commond video file    
            
            if not os.path.isfile(pathfile):
                print("Input video file ", pathfile, " doesn't exist")
                usage()
                sys.exit()
            
            filename, file_extension = os.path.splitext(pathfile)
            print('filename: ', filename)
            print('file_extension: ', file_extension)
            
            output = 'final' + file_extension
            clip = VideoFileClip(pathfile)
            # Print the video duration
            print('Duration: ',  clip.duration, 's')
            
            # other way to computer the total number of frames
            # frames = int(clip.fps * clip.duration)
            # print('n° of frames: ', frames)
            # Print the video clip size
            print('Size: ', clip.size)
            
            if(clip.size != (1280,720)):
                print('Resizing video frame...')
                clip = clip.resize((1280, 720))
                print('Resized Size: ', clip.size)
            
            # Start timer
            start = timeit.default_timer()
            
            video_clip = clip.fl_image(process_image)
            
            video_clip.write_videofile(output, audio=False)
            tem = timeit.default_timer() - start
            t = str(datetime.timedelta(seconds=tem))
            # Print the processing time
            print('Total Processing Time: ', t, '[h:m:s]')
            
            print("Number of total processed frames: ", processed_frames)
            print('Total number of bad detections: ', bad_frame)
            print('Total number of good detections: ', processed_frames-bad_frame)
            print("Finish Processing...")
            
        else :
            print("Unable to read file, format file not supported yet. Exiting...")
            usage()
            sys.exit()

if __name__ == '__main__':
    argparser = argparse.ArgumentParser(description='Advanced Lane Line Detection')
    requiredNamed = argparser.add_argument_group('required named arguments')
    requiredNamed.add_argument("-i", "--input", required=True, help="path to input image or video")
    args = vars(argparser.parse_args())
    
    main(args)
